#!/bin/sh
#
# Rcs_ID="$RCSfile: oraback.sh,v $"
# Rcs_ID="$Revision: 1.49 $ $Date: 2000/12/14 09:29:02 $"
#
# @(#)oraback.sh  -  (Hot backups for Oracle databases)
# Copyright (C) 1996 Curtis Preston - curtis@backupcentral.com
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 2 of the License, or (at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
# more details.
#
# For a copy of the license, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139
#
# Acknowledgements to: Samuel_Binder@coretech-group.com
# (For assistance with sql commands and early coding.)
#
#############################################################################
##Site-specific section (change as appopriate)

DEBUG=N                  #Set this to "Y" turn on set -x for all functions
COMPRESS=compress        #Can be set to gzip if you have it
Z='Z'                    #Compression suffix (e.g. Z for compress, gz for gzip)
BINDIR=/usr/local/bin            #Location of this and related programs
ORACLE=oracle            #ID that script will run as
DBAGROUP=dba             #GROUP that should own backup directory
ORATAB=/var/opt/oracle/oratab #Location of oracle's oratab

#ORACLE_HOME is normally determined by oratab, but you can set it here
ORACLE_HOME=`grep -v '^#' $ORATAB|awk -F':' '{print $2}' |tail -1`

ORACONF=/usr/local/bin/oraback.conf    #Oraback.sh config file location
TMP=/var/tmp                     #Where temporary and permanent logs are kept
SUCCESSMAIL=Y            #Set to N if you only want to be mailed on failure
PERMLOG=/db/oracle/oraback.log         #Full path of oraback.sh log

PATH=$PATH:/usr/bin:/usr/sbin:/sbin:/usr/5bin:/bin:$BINDIR
ORIG_PATH=$PATH

#If you've changed the default init.ora and config.ora locations,
#change them here.
Find_config_files() {
PFILE=$ORACLE_HOME/dbs/init${ORACLE_SID}.ora
IFILE=$ORACLE_HOME/dbs/config${ORACLE_SID}.ora
TNSNAMES=$ORACLE_HOME/network/admin/tnsnames.ora
#SQLNET=$ORACLE_HOME/network/admin/sqlnet.ora
LISTENER=$ORACLE_HOME/network/admin/listener.ora
ALERT=$ORACLE_BASE/admin/${ORACLE_SID}/bdump/alert${ORACLE_SID}.log
}

Preshut() {        #Run prior to shutting down instance
[ "$DEBUG" = Y ] && set -x
}

Poststart() {           #Run after starting up instance
[ "$DEBUG" = Y ] && set -x
}

Preback() {             #Run prior to backup
[ "$DEBUG" = Y ] && set -x
}

Postback() {          #Run after entire backup finishes
[ "$DEBUG" = Y ] && set -x
}

export BINDIR ORATAB ORACONF TMP PATH ORIG_PATH

##End site-specific configuration section
#############################################################################

[ "$DEBUG" = Y ] && set -x

Usage() {
 [ "$DEBUG" = Y ] && set -x
 echo "Usage:   $0 [at|\$ORACLE_SID(s)]"
 echo "(Must be run as one of the following users:"
 echo "${USERS}."
 exit 1
}

Log_error() {            #Sends errors to various places
 [ "$DEBUG" = Y ] && set -x
 if [ "$1" -gt 0 ] ; then
  TIME="`date '+%b %e %T'` $HOST"
  MSG=" `basename $0` $ORACLE_SID: WARNING:"
  ERR=`echo $2|sed "s/^/$TIME $MSG /"`
  echo "$ERR" |$TLOG
  echo "$ERR" >>$TMP/$X.Mail
  [ "$1" = 99 ] && touch $TMP/$X.Fatal
  if [ "$3" = exit ] ; then
   Mail_dba
   exit 1
  fi
 fi
}

Error_chk() {     #Check sqldba output files for errors
 [ "$DEBUG" = Y ] && set -x
 ERROR=`egrep -i 'dba\-|ora\-|mgr\-|error' $1`
 [ -n "$ERROR" ] && Log_error 1 "$ERROR"
 [ ! -s $1 ] && Log_error 99 "Bus error on svrmgr"
}

Backup_offline() {             #Backup instance offline
 [ "$DEBUG" = Y ] && set -x
 echo "  Backing up $ORACLE_SID OFFLINE." |$TLOG
 BTYPE=cold ; export BTYPE
 Backup_instance
}

Backup_instance() {     #Backup all tablespaces in an instance
 [ "$DEBUG" = Y ] && set -x
 awk -F':' '{print $1}' $CWD/$TBLSPACE_LIST | sort -u |\
 while read TBS ; do
  export TBS
  if [ -n "$SIMULT" ] ; then            #Do in background if doing simult. b/u
    Throttle                            #(if there's not too many already)
    Backup_one_tablespace &
  else
    Backup_one_tablespace
  fi
 done                                   #Done with all tablespaces

 while [ -n "`ls $TMP/$X.$ORACLE_SID.*.* 2>/dev/null`" ] ; do
   sleep 10               #Wait for all TBS's to finish
 done

}

Backup_one_tablespace() {         #Backup all files in one tbs
 [ "$DEBUG" = Y ] && set -x

 TBS_STAT=`grep "^$TBS:" $CWD/$TBLSPACE_LIST | awk -F':' '{print $4}'|head -1`

                                         #If hot b/up, put in b/u mode
 if [ $BTYPE = "hot" -a "$TBS_STAT" != "READONLY" ] ; then 
  $SVRMGR<<EOF_BOT >$TMP/$X.$TBS.tablebu 2>&1
connect internal;
alter tablespace $TBS begin backup;
exit;
EOF_BOT
  Error_chk $TMP/$X.$TBS.tablebu

  #Another all new section to make sure that they go into backup mode

  #First, we'll get a list of all the filenumbers that are in this tbs
  FILELIST=`grep "^${TBS}:" $CWD/$TBLSPACE_LIST|awk -F: '{print $3}'`
  #Set this to a "bad" value
  BACKUPSTAT=1

  while [ $BACKUPSTAT -eq 1 ] ; do

   #Reset it to a "good" value
   BACKUPSTAT=0

   #Get the backup status of all the datafiles
   $SVRMGR<<EOF_BU >$TMP/$X.$TBS.backup 2>&1
connect internal;
select file#, status from v\$backup;
EOF_BU

   #Now check the backup status of each file
   for FILENUM in $FILELIST ; do

    #The file should say "ACTIVE," not "NOT ACTIVE" if it's in b/u mode
    FILESTAT=`grep "^  *${FILENUM}  *NOT ACTIVE" $TMP/$X.$TBS.backup`

    #If the result is non-null, then the file is NOT in backup mode
    #Set the "retry" flag and break out of this for loop.  (No sense
    #in looking any further.)
    if [ -n "$FILESTAT" ] ; then
     BACKUPSTAT=1
     break
    fi

   done

  done
  #If we've made it this far, all files should be in b/u mode

  echo "  `date +'%H:%M:%S %d/%m/%y'` - Started Tablespace backup for $TBS" \
   | $TLOG
 else
  echo "  Tablespace $TBS is READ-ONLY. It will not be put in backup mode." \
   | $TLOG
 fi

 grep "^$TBS:" $CWD/$TBLSPACE_LIST | awk -F':' '{print $2}' | \
 while read FILE ; do

  DBF_BASENAME=`echo $FILE|sed 's-/-_-g'|sed 's/_//'`
  export DBF_BASENAME

  if [ -n "$SIMULT" ] ; then      #More simult. copying
    export FILE
    Throttle                             #Limit no. of simult. copies
    Copy_one_database_file &
  else
    CT=`cat $TMP/$X.CT`
    echo `expr $CT + 1` > $TMP/$X.CT
    export FILE CT
    Copy_one_database_file               #Done if foreground
  fi
 done                       #Done with all files in TBS

 #As long as there are $X.*.$TBS.* files, this tbs is not done

 while [ -n "`ls $TMP/$X.*.$TBS.files.* 2>/dev/null`" ] ; do
  sleep 10                  #Wait for all files to copy
 done

 #If this was a hot backup, take out of backup mode 
 if [ $BTYPE = "hot" -a "$TBS_STAT" != "READONLY" ] ; then
  $SVRMGR << EOF_TBS > $TMP/$X.$TBS.tablebu 2>&1
connect internal;
alter tablespace $TBS end backup;
exit;
EOF_TBS
  Error_chk $TMP/$X.$TBS.tablebu

  #Another all new section to make sure that they come out of backup mode

  #First, we'll get a list of all the filenumbers that are in this tbs
  FILELIST=`grep "^${TBS}:" $CWD/$TBLSPACE_LIST|awk -F: '{print $3}'`
  #Set this to a "bad" value
  BACKUPSTAT=1

  while [ $BACKUPSTAT -eq 1 ] ; do

   #Reset it to a "good" value
   BACKUPSTAT=0

   #Get the backup status of all the datafiles
   $SVRMGR<<EOF_BU >$TMP/$X.$TBS.backup 2>&1
connect internal;
select file#, status from v\$backup;
EOF_BU

   #Now check the backup status of each file
   for FILENUM in $FILELIST ; do

    #The file should say "NOT ACTIVE," if it's out of b/u mode
    FILESTAT=`grep "^  *${FILENUM}  *NOT ACTIVE" $TMP/$X.$TBS.backup`

    #If the result is null, then the file is still in backup mode
    #Set the "retry" flag and break out of this for loop.  (No sense
    #in looking any further.)
    if [ -z "$FILESTAT" ] ; then
     BACKUPSTAT=1
     break
    fi

   done

  done

 echo "  `date +'%H:%M:%S %d/%m/%y'` - Finished Tablespace backup for $TBS"\
  | $TLOG

fi
}

Copy_one_database_file() {

 [ "$DEBUG" = Y ] && set -x

 #Create file that tells oraback.sh that a copy of this file is running
 touch $TMP/$X.$ORACLE_SID.$TBS.files.$DBF_BASENAME

 [ -n "$TAPE" ] && DEVICE=$TAPE || DEVICE=$CWD/$DBF_BASENAME

 NEW=`find $FILE -newer $CWD/$DBF_BASENAME.$Z -print 2>/dev/null`

 #If datafile newer than b/u,or if no b/u file,copy it
 if [ $? -gt 0 -o "$NEW" = "$FILE" ]  ; then

  echo "  `date +'%H:%M:%S %d/%m/%y'` - ${CT} Backing up $FILE." |$TLOG

  if [ "$COMP" = Y ]  ; then

   if [ -n "$TAPE" ] ; then
    #If going to tape, use dd and compress it
    dd if=$FILE bs=64k 2>/dev/null| $COMPRESS -c >$DEVICE
   else
    #If going to disk, compress it to a .Z file
    dd if=$FILE bs=64k 2>/dev/null| $COMPRESS -c >$DEVICE.$Z
   fi

  else
   #If not compressing, just send it to the device
   dd if=$FILE of=$DEVICE bs=64k 2>/dev/null
  fi

  Log_error $? "Error Copying $FILE to $DEVICE"

 else
  echo "  `date +'%H:%M:%S %d/%m/%y'` - $FILE has not changed." |$TLOG
 fi

 #Remove the file that tells it this datafile is still copying
 rm -f $TMP/$X.$ORACLE_SID.$TBS.files.$DBF_BASENAME
}

Switch_logfile_n_backup_controlfile() {
 [ "$DEBUG" = Y ] && set -x
 echo "  -----------------------------------------------------" |$TLOG
 echo "  Creating backup controlfile - $CWD/backup.controlfile" |$TLOG

 #Get tracefile destination dir
 DUMPDEST=`cat $CWD/dumpdest.list`

 #Create a file to be used to find the b/u cont. file to trace
 touch $DUMPDEST/tracefile.timefile.empty

 [ -f $CWD/backup.controlfile ] && rm -f $CWD/backup.controlfile
 $SVRMGR << EOF_CONTROL >$TMP/$X.control 2>&1
connect internal;
alter system checkpoint;
alter system archive log current;
alter database backup controlfile to '$CWD/backup.controlfile';
alter database backup controlfile to trace;
exit;
EOF_CONTROL
 Error_chk $TMP/$X.control

 #Find most recent archive logs currently in arch_dest
 if [ -d $ARCH_LOG_DIR -a -f $ARCH_LOG_DIR/${ARCH_LOG_FILE}* ] ; then
  NEWER=`find $ARCH_LOG_DIR -type f -name "${ARCH_LOG_FILE}*" \
  -newer $OLD -print`
  NEWER=`echo $NEWER`
  NEWEST=`ls -t $NEWER|head -1`
  OLD_SIZE=`du -s $NEWEST|awk '{print $1}'` ; SLEEP=Y
 else
  [ ! -d $ARCH_LOG_DIR ] \
   && Log_error 1 "Log destination (${ARCH_LOG_DIR}) not found."
  [ ! -f $ARCH_LOG_DIR/${ARCH_LOG_FILE}* ] \
   && Log_error 1 "NO archive logs found in $ARCH_LOG_DIR."
 fi
}

Copy_control_and_essential_redolog_files() {
 [ "$DEBUG" = Y ] && set -x

#This part is essentially worthless, as far as most dbas are concerned,
#but is provided by request of some oraback.sh users.
#(I think it helps in a cold backup, anyway...)

 if [ -f $CWD/$CONTROLFILE_LIST ] ; then


 echo "  Manually copying controlfiles:  " | $TLOG
 cat $CWD/$CONTROLFILE_LIST |sed 's/^/   /' | sed 's/  *$//' | $TLOG

  for CONTROLFILE in `grep -v '^$' $CWD/$CONTROLFILE_LIST`
  do
   CONTROLFILE_BASENAME=`echo $CONTROLFILE|sed 's-/-_-g'|sed 's/_//'`
   $COMPRESS -c $CONTROLFILE > $CWD/$CONTROLFILE_BASENAME.$Z
   Log_error $? "Error copying $FILE to $CWD"
  done

 else
   Log_error 1 "$CWD/$CONTROLFILE_LIST: File not found"
 fi

 if [ -f $CWD/$LOGFILE_LIST ] ; then
  if [ -z "$NOLOG" ] ; then

    #If Instance is up, remove old logs in b/u dir, otherwise leave them
    if [ "$DB_UP" = 2 ] ; then

     REMOVE=`ls $CWD/backup.${ARCH_LOG_FILE}* 2>/dev/null|grep -v "$BOLD"`

     if [ -n "$REMOVE" ] ; then
      echo "  Removing old archive logs: " |$TLOG
      ls $CWD/backup.${ARCH_LOG_FILE}* 2>/dev/null|grep -v "$BOLD" \
       |sed 's/^/   /' | $TLOG
      rm -f $REMOVE
     else
      echo "  NO old archive logs found." |$TLOG
     fi

    fi

    echo "  Copying archived redo logs created since $0 started:" |$TLOG
    find $ARCH_LOG_DIR -type f -name "$ARCH_LOG_FILE*" -newer $OLD -print \
     >$TMP/$X.newer
    NEWER=`cat $TMP/$X.newer`

    if [ -n "$NEWER" ] ; then
     echo "  New archive logs found: " |$TLOG
     cat $TMP/$X.newer|sed 's/^/   /' | $TLOG

     for ARCHIVED_LOG_FILE in $NEWER ; do
      ARCHIVED_LOG_FILE_BASENAME=`basename $ARCHIVED_LOG_FILE`
      BACKUP_COPY_NAME="backup.$ARCHIVED_LOG_FILE_BASENAME"
      cp -p $ARCHIVED_LOG_FILE $CWD/$BACKUP_COPY_NAME
      [ $? -ne 0 ] && Log_error 1 "Error copying $ARCHIVED_LOG_FILE to $CWD"
     done

     COMPRESS_LIST=`ls $CWD/backup.${ARCH_LOG_FILE}* |egrep -v '\.gz$|\.z$|\.Z$'`
     $COMPRESS -f $COMPRESS_LIST 2>/dev/null
     [ $? -ne 0 ] && Log_error 1 "Error compressing $CWD/${ARCH_LOG_FILE}\*"

    else
     echo "  NO archive logs made since this backup started." |$TLOG
    fi

  else
    echo "  No logs:  $ORACLE_SID set to NOARCHIVELOG." |$TLOG
  fi
 else
    Log_error 1 "$CWD/$LOGFILE_LIST: File not found"
 fi

 if [ -f $CWD/$DUMPDEST_LIST ] ; then
  echo "  Creating controlfile trace file:
   $CWD/recreate-controlfile.txt" |$TLOG
  DUMPDEST=`cat $CWD/$DUMPDEST_LIST|sed "s-\?-$ORACLE_HOME-"`
  #Find tracefile and copy it
  
  TRACEFILE=`find $DUMPDEST -newer $DUMPDEST/tracefile.timefile.empty \
   -type f -print|tail -1`
  Log_error $? "Did not find a tracefile in $DUMPDEST."
  cp $TRACEFILE $CWD/recreate-controlfile.txt
  Log_error $? "Could not copy $TRACEFILE to $CWD/recreate-controlfile.txt."
  rm -f $DUMPDEST/tracefile.timefile.empty
  Log_error $? "Could not remove $CWD/recreate-controlfile.txt."
 else
    Log_error 1 "$CWD/$DUMPDEST_LIST: File not found"
 fi


 if [ -n "$TAPE" ] ; then      #Send disk b/u's to tape
   CT=`cat $TMP/$X.CT`
   echo `expr $CT + 1` > $TMP/$X.CT
   echo "  ${CT}- Tar file of $CWD." |$TLOG
   cd $CWD
   tar cvf $TAPE ./*
  fi
}

Copy_online_redologs() {
 [ "$DEBUG" = Y ] && set -x

  echo "  Copying online redo logs:  " |$TLOG
  cat $CWD/$REDOLOGS_LIST |sed 's/^/   /' |sed 's/  *$//'| $TLOG
  
  cat $CWD/$REDOLOGS_LIST | \
  while read REDOLOG
  do
    LOGBASE=`echo $REDOLOG|sed 's-/-_-g'|sed 's/_//'`
    dd if=$REDOLOG bs=64k 2>/dev/null| $COMPRESS -c >$CWD/$LOGBASE.$Z
    [ $? -ne 0 ] && Log_error 1 "Error compressing $REDOLOG to $CWD/LOGBASE.$Z"
  done

}

Shutdown_instance() {
 [ "$DEBUG" = Y ] && set -x
 echo "  (Shutting down $ORACLE_SID `date` )" |$TLOG

 Preshut                         #Pre-Shutdown function

#I get a lot of questions about the following section.  This way of shutting
#down is to ease the concerns of those who believe that a shutdown immediate
#is "harsh."  This is what Oracle support advised me to do.  Shutdown immed.,
#startup restrict, then shutdown normal.  The sleeps are just to make
#sure that it's completely done before starting up/shutting down again.

 $SVRMGR<<EOF_SID >$TMP/$ORACLE_SID.shut.out 2>&1
connect internal;
shutdown immediate;
host sleep 2
startup restrict;
host sleep 2
shutdown;
exit;
EOF_SID
 Error_chk $TMP/$ORACLE_SID.shut.out
}

Startup_instance() {
 [ "$DEBUG" = Y ] && set -x
 echo "  (Starting up $ORACLE_SID `date` )" |$TLOG
 $SVRMGR<<EOF_STARTUP >$TMP/$ORACLE_SID.start.out 2>&1
connect internal;
startup;
exit;
EOF_STARTUP
 Error_chk $TMP/$ORACLE_SID.start.out
 Poststart                       #Post-startup function
}

Throttle() {

#This function is used to make sure that no more that $SIMULT datafiles
#are being backed up at once.

 [ "$DEBUG" = Y ] && set -x
 sleep 2
 DDNUM=`ls $TMP/$X.*.*.files.* 2>/dev/null|wc -l|sed 's/ //g'`
 while [ "$DDNUM" -ge $SIMULT ]
 do
  #Sleep if there are enough files being copied
  sleep 10
  DDNUM=`ls $TMP/$X.*.*.files.* 2>/dev/null|wc -l|sed 's/ //g'`
 done
}

Mail_dba() {
 [ "$DEBUG" = Y ] && set -x

 #$X.Mail will only have stuff in it if there were errors somewhere

 if [ -s $TMP/$X.Mail ] ; then

  #If there were errors, mail them to the dba
  $L_MAIL $L_S "$MESS" $DBAS < $TMP/$X.Mail

 else

 #If there were no errors, check SUCCESSMAIL variable
  #If SUCCESSMAIL is set to Y, then notify them of successful backup
   if [ "$SUCCESSMAIL" = Y ] ; then
    $L_MAIL $L_S "Oracle backup on $HOST successful" $DBAS < $LOG
   fi
 fi

}

Copy_config_files() {

if [ -n "$IFILE" ] ; then
 if [ -f "$IFILE" ] ; then
  echo "  Backing up $IFILE" | $TLOG
  cp $IFILE $CWD
 else
  Log_error 1 "Ifile (${IFILE}) not found."
 fi
fi

if [ -n "$PFILE" ] ; then
 if [ -f "$PFILE" ] ; then
  echo "  Backing up $PFILE " | $TLOG
  cp $PFILE $CWD
 else
  Log_error 1 "Pfile (${PFILE}) not found."
 fi
fi

if [ -n "$TNSNAMES" ] ; then
 if [ -f $TNSNAMES ] ; then
  echo "  Backing up $TNSNAMES" | $TLOG
  cp $TNSNAMES $CWD
 else
  Log_error 1 "Pfile (${TNSNAMES}) not found."
 fi
fi

if [ -n "$SQLNET" ] ; then
 if [ -f $SQLNET ] ; then
  echo "  Backing up $SQLNET" | $TLOG
  cp $SQLNET $CWD
 else
  Log_error 1 "Pfile (${SQLNET}) not found."
 fi
fi

if [ -n "$LISTENER" ] ; then
 if [ -f $LISTENER ] ; then
  echo "  Backing up $LISTENER" | $TLOG
  cp $LISTENER $CWD
 else
  Log_error 1 "Pfile (${LISTENER}) not found."
 fi
fi
}

Check_bkmode() {
 [ "$DEBUG" = Y ] && set -x

$SVRMGR << EOF > $TMP/$X.ckbkmode 2>&1
connect internal;
select * from v\$backup where status = 'ACTIVE';
exit;
EOF
 ACTIVE=`awk '{print $2}' $TMP/$X.ckbkmode|grep ACTIVE`
 [ -z "$ACTIVE" ] || Log_error 1 "Some files are still in backup mode!"
}

# Check to see if this program was called by another, and should not rewind the tape

if [ "$1" = norewind ] ; then
      NOREWIND=Y
	shift
fi

#Setup log and global variables in case localpath.sh barfs
L_MAIL=mail ; L_S='' ; MESS='' ; export L_MAIL L_S MESS

if [ -f $BINDIR/config.guess ] ; then
 LOCAL_OSnR=`$BINDIR/config.guess`
 export LOCAL_OSnR
else
 Log_error 1 "NO $BINDIR/config.guess - ABORTING BACKUP" exit
fi

. $BINDIR/localpath.sh

MESS="ORACLE BACKUP ERROR!" ; export MESS

if [ ! -s $ORATAB ]; then
 Log_error 1 "NO $ORATAB - ABORTING BACKUP" exit
fi

LOG=$PERMLOG.tmp

cp /dev/null $LOG
Log_error $? "Unable to create $LOG" exit

chown $ORACLE $LOG
Log_error $? "Unable to chown owner to $ORACLE $LOG" exit

TLOG="tee -a $LOG"
X=$$
LOGFILE_LIST=logfile.list
CONTROLFILE_LIST=controlfile.list
REDOLOGS_LIST=redologs.list
TBLSPACE_LIST=tablespace.list
DUMPDEST_LIST=dumpdest.list
export X LOGFILE_LIST CONTROLFILE_LIST TBLSPACE_LIST REDOLOGS_LIST
export DUMPDEST_LIST

HOST=`uname -n|awk -F'.' '{print $1}'`
[ -n "$HOST" ] || HOST=`hostname|awk -F'.' '{print $1}' `

if [ -z "$HOST" ] ; then
 Log_error 1 "Unable to obtain hostname!" exit

fi

SKIP=`grep "^$HOST.master:" $ORACONF | awk -F':' '{print $2}' `
COLD=`grep "^$HOST.master:" $ORACONF | awk -F':' '{print $3}' `
 [ "$COLD" -gt 0 ] 2>/dev/null  &&  DAY=`date +%d`  ||  DAY=`date +%a`
 [ "$COLD" = '*' ]  &&  COLD=$DAY
 if [ "$DAY" = "$COLD" ] ; then
  TYPE=COLD
  TIME=`grep "^$HOST.master:" $ORACONF  | awk -F':' '{print $4}' `
 else
  TYPE=HOT
  TIME=`grep "^$HOST.master:" $ORACONF  | awk -F':' '{print $5}' `
 fi
TAPE=`grep "^$HOST.master:" $ORACONF  | awk -F':' '{print $6}' `
USERS=`grep "^$HOST.master:" $ORACONF | awk -F':' '{print $7}' `
 [ -z "$USERS" ]  &&  USERS=$ORACLE
SIMULT=`grep "^$HOST.master:" $ORACONF | awk -F':' '{print $8}' `
BACKUPDIR=`grep "^$HOST.master:" $ORACONF |awk -F':' '{print $9}' `
if [ ! -d "$BACKUPDIR" ] ; then
 echo "Backup directory $BACKUPDIR does not exist!  Exiting..." ; exit 1
fi
if [ -n "$TAPE" -a -n "$SIMULT" ] ; then
 echo "Simultaneous backups not supported for" |$TLOG
 echo "tape devices.  (Will do serially.) " |$TLOG
 SIMULT=''  ;  export SIMULT
fi
COMP=`grep "^$HOST.master:" $ORACONF | awk -F':' '{print $10}' `
DBAS=`grep "^$HOST.master:" $ORACONF |awk -F':' '{print $11}' `

#This section is to let users know if they need to change the format
#of $ORACONF, since the MAIL command is now figured out in localpath.sh
if [ -n "`echo $DBAS|grep '[Mm]ailx* '`" ] ; then
 DBAS=`echo $DBAS|sed 's/.* //'`
 echo "
WARNING: You need to change to the format of field 11 in oraback.conf.
It now only needs to contain the mail id's that will get the mail.
It has been changed to $DBAS for this run of $0,
but you will need to change it in $ORACONF
to make the change permanent."

fi

WHO=`id | awk -F'(' '{print $2}' | awk -F')' '{print $1}' `

if [ -z "$BACKUPDIR" -o -z "$DBAS" ] ; then
  echo "Field 9 or 12 in $ORACONF is empty!"   |$TLOG
  echo "(9=BACKUPDIR, 12=complete Mail command)" |$TLOG
  exit 1
fi
                                   #Build list of SID's
if [ $# -ne 0 -a "$1" != "at" ] ; then
 GREP=`echo $* | sed 's/ /:|^/g' | sed 's/$/:/'`
 ORACLE_SID_LIST=`egrep "^$GREP" $ORATAB`
 ARG=backup
else
 ORACLE_SID_LIST=`grep -v '^\#' $ORATAB|grep -v '\*'`
 case $1 in
  query )
   QUERYONLY=Y ; ARG=backup
  ;;
  at )
   ARG=at
  ;;
  * )
   ARG=backup
  ;;
 esac
fi

if [ -z "`echo $WHO | egrep \"$USERS\"`" ] ; then
 echo "WRONG LOGIN!\n"
 Usage
fi

touch $ORACONF ${ORACONF}.2
Log_error $? "Unable to create $ORACONF.2 (Needed 4 SKIP feature.)" exit

                    #Skip b/u if find "skip" in ORACONF
if [ "$SKIP" = "skip" ]; then
 sed "s/$HOST.*:skip:/$HOST.master::/" $ORACONF >${ORACONF}.2
 mv -f ${ORACONF}.2 $ORACONF
 echo "SKIPPING $0 ON $HOST TONIGHT ONLY DUE"   |$TLOG
 echo "TO THE WORD \"skip\" IN $ORACONF!!"      |$TLOG
 cat $LOG >>$PERMLOG
 exit 0
fi

case ${ARG} in

at)

 #Check's $ORACONF for what time the sheduled backup is to run, and then
 #schedules an 'at' job to run the script with no args, which will put it
 #in the 'backup' case of this case statement

 if [ -z "$TIME" ]; then
  Log_error 1 "No backup time found in $ORACONF" exit
  exit 1
 fi

case $LOCAL_OSnR in

 #The -s option is available in most unixes, and specifies to
 #start an at job w/Bourne sh.  If you can't use this, and you
 #are using csh as Oracle's shell, you'll need to change the output
 #redirection in the section that creates the at job
 #(This is one of the many reasons why I hate csh...)

 *hpux*|*irix* ) _S=''   ;;
 * )     _S='-s' ;;
esac

 at $_S $TIME <<EOF_AT          
$BINDIR/oraback.sh > $TMP/oraback.out 2>&1
EOF_AT
;;

backup)

#Actually run backup

 echo "==============================================================" |$TLOG
 echo "Oracle database backup started on $HOST: `date '+%X %m/%d/%Y'`" |$TLOG
 echo "==============================================================" |$TLOG
 echo "1" > $TMP/$X.CT
 Preback                           #Pre-backup function

 cd $TMP
 LABEL="$0 $* `date +%m/%d/%Y`"
 LABEL=`echo $LABEL| sed 's/ /_/g'|sed 's-/-_-g'`

 echo |tee -a $LOG
 echo "Creating table of contents for this backup..." |tee -a $LOG
 if [ -n "$TAPE" ] ; then
  echo "Then placing it as a file called $LABEL on $DEVICE" |tee -a $LOG
  echo "(and verifying that $DEVICE is set to NO-REW...)" |tee -a $LOG
 fi
 echo |tee -a $LOG

cat >$LABEL <<EOF_HEAD

----------------------------------------------------------------------
This tape is a full backup of the Oracle instances on $HOST made `date`
The following is a table of contents of this backup.
(This information is most useful if the backup is sent to tape.)

It is in the following format:

 INSTANCE:ORACLE_SID:ORACLE_HOME_DIR:DATABASE_STATUS (in a 3 letter code)
 -------------------
 ORACLE_SID:FILE_NUMBER:TABLESPACE_NAME:DATA_FILE_NAME
 ORACLE_SID:FILE_NUMBER:TABLESPACE_NAME:DATA_FILE_NAME

 INSTANCE:ORACLE_SID:ORACLE_HOME_DIR:DATABASE_STATUS (in a 3 letter code)
 -------------------
 ORACLE_SID:FILE_NUMBER:TABLESPACE_NAME:DATA_FILE_NAME
 ORACLE_SID:FILE_NUMBER:TABLESPACE_NAME:DATA_FILE_NAME

To recover any of the DATA_FILEs:
1. cd to the directory that should contain that file.
2. $L_MT $L_F $DEVICE fsf FILE_NUMBER
3. dd if=$DEVICE bs=64k of=DATA_FILE_NAME

WARNING!  Databases can go up and down in the middle of a backup!
That will drastically affect what gets put on this tape!
This table of contents represents only what SHOULD be put on the tape,
based on the DB_STAT information at the time of label creation.
If there is any question as to what actually went TO the tape, the log
of this session will be appended to this tape as a tar file.
To read that tar file follow these steps:

1. Look at the highest FILE_NUMBER listed
2. $L_MT $L_F $DEVICE fsf (FILE_NUMBER + 1)
3. tar xvf $LOG

If this works, then all the files went to the tape.  If it does not work,
it means that some did not go to the tape because a database was down
when it should have been up, or some other reason.  If that is the case, then
use the following loop:

COUNT=1
while true
do
  tar xvf $DEVICE
   if [ \$? -eq 0 ] ; then
     exit
    else
     $L_MT $L_F $DEVICE  $L_REWIND
     $L_MT $L_F $DEVICE  fsf \$COUNT
     COUNT=\`expr $COUNT + 1\`
   fi
done

ALSO, this program now supports a no-rewind option, which allows it to be
called by another program (e.g. hostdump.sh).  If called with this argument,
the tape will not start at the beginning.  This means that the tape file
that will have the tar file will be unpredictable.

----------------------------------------------------------------------

The proposed table of contents for this backup:

EOF_HEAD

FILE_NUMBER=1           #Initialize the counter
                        #For each SID we are backing up
 for LINE in $ORACLE_SID_LIST
 do
  ORACLE_SID=`echo $LINE  | awk -F':' '{print $1}' `
  CWD=$BACKUPDIR/$ORACLE_SID
  ORACLE_HOME=`echo $LINE | awk -F':' '{print $2}' `

  Find_config_files

  #Try to find the appropriate sql command to use
  [ -f $ORACLE_HOME/bin/sqldba ]   && SVRMGR=sqldba
  [ -f $ORACLE_HOME/bin/svrmgrl ]  && SVRMGR=svrmgrl
  [ -f $ORACLE_HOME/bin/svrmgr30 ] && SVRMGR=svrmgr30

 if [ -z "$SVRMGR" ] ; then
     Log_error 1 "$ORACLE_SID has an invalid working directory!"
     continue
  fi
 
  export CWD ORACLE_SID ORACLE_HOME SVRMGR
  mkdir $CWD 2>/dev/null

  echo " Checking the status of Oracle instance $ORACLE_SID.."|$TLOG

  chmod 775 $CWD  ;  chgrp $DBAGROUP $CWD

  PATH=$ORIG_PATH:$ORACLE_HOME/bin  ;  export PATH

  #This command will only work if the database is up
  #and that's exactly what I'm using it for, to see if it's up

  $SVRMGR << EOF_UP >$TMP/$X.databaseup 2>&1
connect internal;
select * from sys.dba_data_files;
exit;
EOF_UP
  Error_chk $TMP/$X.databaseup

  DBUP=`grep -i '[1-9][0-9]* rows selected\.' $TMP/$X.databaseup`

  if [ -n "$DBUP" ] ; then            #Database is UP

   DB_UP=2
   echo "  Oracle instance $ORACLE_SID appears to be running..."|$TLOG

   #Make list of tablespaces
   $SVRMGR <<EOF_XTBS >$TMP/$X.tbs.tmp
connect internal;
select tablespace_name, file_name, file_id from sys.dba_data_files;
exit;
EOF_XTBS

   awk 'NF == 3 && $2 ~ /\// {print $1":"$2":"$3}' $TMP/$X.tbs.tmp \
    > $TMP/$X.tbs
   Error_chk $TMP/$X.tbs.tmp

   #Find out which datafiles are read-write and read-only
   $SVRMGR <<EOF_TB_READ >$TMP/$X.tbs.wr
connect internal;
select name, enabled from v\$datafile;
exit;
EOF_TB_READ

   #Add this to the tablespace.list file
   cat $TMP/$X.tbs|while read LINE
   do
    FILE=`echo $LINE|awk -F: '{print $2}'`
    FILENUM=`echo $LINE|awk -F: '{print $3}'`
    STATUS=`grep "${FILE}.*WRITE" $TMP/$X.tbs.wr`
    [ -n "$STATUS" ] && STATUS=READWRITE || STATUS=READONLY
    echo "$LINE:$STATUS"
   done | sort >$CWD/$TBLSPACE_LIST

   #Get names of online redologs
   $SVRMGR <<EOF_LOGFILE > $TMP/$X.rdl
connect internal;
select member from v\$logfile ;
exit;
EOF_LOGFILE

cp $TMP/$X.rdl $TMP/backup.$X.rdl

    egrep '/|\\' $TMP/$X.rdl |grep -v "PL.SQL" > $CWD/$REDOLOGS_LIST

   Error_chk $TMP/$X.rdl

   #List archiving info
   $SVRMGR <<EOF_ARCHIVE >$TMP/$X.lf
connect internal;
archive log list;
exit;
EOF_ARCHIVE
   awk '/^Ar/ {print $3}' $TMP/$X.lf |sed "s-\?-$ORACLE_HOME-" \
    > $CWD/$LOGFILE_LIST
   Error_chk $TMP/$X.lf

   LOGMODE=`egrep -i ' log mode.*[^o] archive' $TMP/$X.lf`

   #The above grep pattern will match 'ARCHIVELOG'(V7) and 'Archive Mode' (V8)
   #But will NOT match "NOARCHIVELOG"(V7) or "No Archive Mode" (V8)

   if [ -n "$LOGMODE" ] ; then
    LOGS_UP=20
    echo "  Oracle instance $ORACLE_SID is in ARCHIVELOG mode..."|$TLOG
   else
    LOGS_UP=10
    echo "  Oracle instance $ORACLE_SID is NOT in ARCHIVELOG mode..."|$TLOG
   fi

    #Get names of controlfiles
    $SVRMGR <<EOF >$TMP/$X.cf
connect internal;
select name from v\$controlfile;
exit;
EOF
   grep "^/" $TMP/$X.cf > $CWD/$CONTROLFILE_LIST

   Error_chk $TMP/$X.cf

   #Get v$parameter data
   $SVRMGR <<EOF |sed "s-\?-$ORACLE_HOME-" >$TMP/$X.parameter
connect internal;
select name, value from v\$parameter;
exit;
EOF

   grep user_dump_dest $TMP/$X.parameter |awk '{print $NF}' > $CWD/$DUMPDEST_LIST
   Error_chk $TMP/$X.parameter

  else                           #else Database is DOWN
   DB_UP=1  ;  LOGS_UP=10
   echo "  Oracle instance $ORACLE_SID appears to be down..."|$TLOG
  fi

  LOGDEST=`grep -v '^$' $CWD/$LOGFILE_LIST`

  #LOGDEST could be a directory, or a dir with a file pattern tacked
  #at the end.  If it has a file pattern, delete it and leave dir.

  if [ ! -d "$LOGDEST" ] ; then
   ARCH_LOG_DIR=`echo $LOGDEST | sed 's-/[^/]*$--'`
   ARCH_LOG_FILE=`basename $LOGDEST`
  else
   ARCH_LOG_DIR=$LOGDEST

   #Get archive format from v$parameter data obtained above
   ARCH_LOG_FILE=`grep log_archive_format $TMP/$X.parameter \
    |awk '{print $2}'|sed 's/%./*/g'`
  fi

  OLD=`ls -t $ARCH_LOG_DIR/${ARCH_LOG_FILE}* 2>/dev/null| head -3 | tail -1`
  if [ -z "$OLD" ] ; then
   echo "  You don't seem to have any archivelogs in $ARCH_LOG_DIR.
   (This should be no cause for alarm if you are automatically moving them elsewhere.)"|$TLOG
   touch $ARCH_LOG_DIR/fake_old_logfile
   OLD="$ARCH_LOG_DIR/fake_old_logfile"
  fi
  BOLD=`basename $OLD`

  NOLOG=`grep "^$HOST:$ORACLE_SID:NOARCHIVELOG" $ORACONF`
  NOHOTBACKUP=`grep -i "$HOST:$ORACLE_SID:.*nohotbackup" $ORACONF`

  if [ -n "$NOHOTBACKUP" ]  ; then
    HOTBACKUP=100
    echo "  Oracle instance $ORACLE_SID is NOT supposed to backed up live." \
     |$TLOG
  else
    HOTBACKUP=200
    echo "  Oracle instance $ORACLE_SID is supposed to backed up live..."     \
     |$TLOG
  fi

  DB_STAT=`expr $HOTBACKUP + $LOGS_UP + $DB_UP`
  [ -f $TMP/$X.Fatal ]  &&  DB_STAT=221

  if [ "$TYPE" = COLD ] ; then

   #IF instance is not already down, and its cold backup
   #time, then shut it down, else set a variable to not
   #start it when we are done

   case $DB_STAT in
    211|111|221) DONT_START=DONT ;;
    * )
     Switch_logfile_n_backup_controlfile #Function
     Shutdown_instance                   #Function
     DB_UP=1  ;  LOGS_UP=10
     DB_STAT=`expr $HOTBACKUP + $LOGS_UP + $DB_UP`
     [ -f $TMP/$X.Fatal ]  &&  DB_STAT=221
    ;;
   esac
  fi

  case $DB_STAT in
  222)     #DB=up, LOGS=on, Backup gets green light
           INCLUDED=YES
           echo "  Oracle instance $ORACLE_SID WILL receive a hot backup..." \
            |$TLOG
           echo "  Backing up to directory $CWD on $HOST" |$TLOG

           echo "  Table of contents:" |$TLOG  ;;
  211|111) #Database is completely down. $0 can back it up if config files
           #are already in place from a previous run of $0

           if [ -f $CWD/$TBLSPACE_LIST ] ; then
            INCLUDED=YES
            echo "  Oracle instance $ORACLE_SID WILL receive a cold backup..."
            echo "  Table of contents:" |$TLOG
           else
            echo "  Oracle instance $ORACLE_SID WILL NOT be backed up..."
            echo " (It is down and there are no backup config files in"  |$TLOG
            echo " $CWD from a previous run of $0)"  |$TLOG
            INCLUDED=NO
           fi
  ;;
  212)      #Database is up, but is set to NOARCHIVELOG

   if [ -n "$NOLOG" ]  ; then       #Logs should be off
    if [ -n "`echo "$NOLOG"|grep ':offline'`" ] ; then
     #If the dba put "offline" in the $ORACONF,
     #Then it's supposed to be shutdown for a cold backup every time
     INCLUDED=YES
     echo "  Oracle instance $ORACLE_SID has this custom line in $ORACONF:"
     echo "  $NOLOG"
     echo "  The word ":offline:" above specifies that this instance"    |$TLOG
     echo "  is always supposed to receive a cold backup.  It will "     |$TLOG
     echo "  therefore be shutdown for this backup."
     echo "  Table of contents:" |$TLOG
    else
     #Otherwise, logs are off, we know it, and we don't want to shut it down
     INCLUDED=NO
     echo "  Oracle instance $ORACLE_SID is not in ARCHIVELOG mode, and" \
      |$TLOG
     echo "  has a custom line in $ORACONF.  It cannot"                  |$TLOG
     echo "  be backed up live without ARCHIVELOG,"                      |$TLOG
     echo " so it will NOT be backed up."                                |$TLOG
     echo "  If you want it to be backed up offline, you must put"       |$TLOG
     echo "  the following line in $ORACONF:"                            |$TLOG
     echo "  $HOST:$ORACLE_SID:NOARCHIVELOG:offline"                     |$TLOG
    fi
   else
     #Logs are off, and no special flags have been placed in $ORACONF
     #Maybe the dba doesn't know... Tell the dba loudly!
     Log_error 1 "CANNOT BACKUP $ORACLE_SID (NOARCHIVELOG)!"
     INCLUDED=NO
     echo "  Oracle instance $ORACLE_SID is not in ARCHIVELOG mode, and" |$TLOG
     echo "  has NO custom line in $ORACONF.  It cannot"                 |$TLOG
     echo "  be backed up live without ARCHIVELOG, "                     |$TLOG
     echo "  so it will NOT be backed up."                               |$TLOG
     echo "  If you want to back it up only when cold backups run, you"  |$TLOG
     echo "  must put the following line in $ORACONF:"                   |$TLOG
     echo "  $HOST:$ORACLE_SID:NOARCHIVELOG"                             |$TLOG
     echo "  If you want it to be backed up cold every night, you must"  |$TLOG
     echo "  put the following line in $ORACONF:"                        |$TLOG
     echo "  $HOST:$ORACLE_SID:NOARCHIVELOG:offline"                     |$TLOG
   fi
   ;;
   112|122)           #Word 'nohotbackup' is in ORACONF
    echo "  Oracle instance $ORACLE_SID will not be backed up."          |$TLOG
    echo " Found the following line in $ORATAB:"               |$TLOG
    echo " $NOHOTBACKUP"                                        |$TLOG
    echo " (It will only be backed up on cold backup nights"   |$TLOG
    INCLUDED=NO
   ;;
   221)            #Fatal error in early sqldba command
    echo "Something REALLY bad happened!" |$TLOG
    INCLUDED=NO
   ;;
  esac

  if [ "$INCLUDED" = YES ] ; then
   echo "INSTANCE:$ORACLE_SID:$ORACLE_HOME:$DB_STAT" |tee -a $LABEL

   #Now a for loop to create a list that looks like:
   #ORACLE_SID:FILE_NUMBER:TABLESPACE:DATA_FILE

   for TABLESPACEnDATA_FILE in `cat $CWD/$TBLSPACE_LIST`
   do
    echo "$ORACLE_SID:$FILE_NUMBER:$TABLESPACEnDATA_FILE" |tee -a $LABEL
    FILE_NUMBER=`expr $FILE_NUMBER + 1`
   done
   echo "$ORACLE_SID:$FILE_NUMBER:Redologs (and other logs) in one tar file" \
    |tee -a $LABEL
   FILE_NUMBER=`expr $FILE_NUMBER + 1`
  fi

 done

if [ -n "$TAPE" ]; then         #If using tape, label

 if [ "$NOREWIND" != "Y" ] ; then
  $L_MT $L_F $TAPE $L_REWIND
 else
  echo "Starting in the middle of the tape (no rewind argument used)." | $TLOG
 fi

 if [ $? -eq 0 ]; then
  sleep 2
  tar cf $TAPE $LABEL
 else
  Log_error 1 "$TAPE failed to rewind!" exit
  exit 1
 fi

 if [ "$NOREWIND" != "Y" ] ; then
  if [ -n "`tar tf $LABEL 2>/dev/null|grep $LABEL`" ]
  then
   Log_error 1 "$TAPE NOT the non-rewind device" exit
   exit 1
  else

   $L_MT $L_F $TAPE $L_REWIND
   $L_MT $L_F $TAPE fsf 1

  fi

  echo "Label the tape in $TAPE: \"$LABEL\"." |$TLOG

 fi

fi #end of label creation and tape setup, if using a tape

for LINE in `grep '^INSTANCE:' $TMP/$LABEL`
do
  #get sid name, status number from label file
  ORACLE_SID=`echo $LINE  | awk -F':' '{print $2}' `
  SID=$ORACLE_SID
  CWD=$BACKUPDIR/$ORACLE_SID
  ORACLE_HOME=`echo $LINE | awk -F':' '{print $3}' `
  DB_STAT=`echo $LINE | awk -F':' '{print $4}' `
  
  Find_config_files

  [ -f $ORACLE_HOME/bin/sqldba ] && SVRMGR=sqldba
  [ -f $ORACLE_HOME/bin/svrmgrl ] && SVRMGR=svrmgrl
  [ -f $ORACLE_HOME/bin/svrmgr30 ] && SVRMGR=svrmgr30

 if [ -z "$SVRMGR" ] ; then
     Log_error 1 "$ORACLE_SID has an invalid working directory!"
     continue
  fi
 
  export CWD ORACLE_SID SID ORACLE_HOME SVRMGR

  echo " -----------------------------------------------------------"| $TLOG
  echo " Backup of $ORACLE_SID started `date +%T` by $LOGNAME " |$TLOG

  case $DB_STAT in
  222)         #DB=up, LOGS=on, Backup gets green light

   BTYPE=hot ; export BTYPE

   if [ "$QUERYONLY" != Y ] ; then
    Backup_instance                     #Function
    Switch_logfile_n_backup_controlfile #Function
    Copy_online_redologs
    Copy_control_and_essential_redolog_files
    Copy_config_files
    Check_bkmode

   fi
   ;;


  211|111)                 #Database is completely down

   if [ $DB_STAT = 211 -a "$TYPE" = "HOT" ] ; then
    Log_error 1 "!!$ORACLE_SID IS DOWN During a Hot Backup!!"
   fi

   if [ "$QUERYONLY" != Y ] ; then
    Backup_offline
    Copy_online_redologs
    Copy_control_and_essential_redolog_files
    Copy_config_files
   fi
   ;;

  212)      #Database is up, but is set to NOARCHIVELOG

    #It should only be in the include list if it was set for an offline
    #Backup, so go ahead an do an offline backup!

    if [ "$QUERYONLY" != Y ] ; then
     Shutdown_instance          #Configured for offline
     Backup_offline                   #backup, so do it
     Copy_online_redologs
     Copy_control_and_essential_redolog_files
     Startup_instance
     Switch_logfile_n_backup_controlfile
     Copy_config_files
     Error_chk $TMP/$X.control
    fi
   ;;
  esac

  #If instance was not already down before the /bu started,
  #then start it up
  if [ "$DONT_START" != DONT ] ; then
   if [ "$QUERYONLY" != Y ] ; then
    [ "$TYPE" = COLD ]  &&  Startup_instance
   fi
  fi

  echo " Backup of $ORACLE_SID ended `date +%T` by $LOGNAME " |$TLOG

 done                              #done with all $ORACLE_SIDS

 Postback       #Run Post-backup script if there is one

 if [ -n "$TAPE" ] ; then      #eject tape
  $L_MT $L_F $TAPE $L_OFFLINE
	if [ $? -gt 0 ] ; then
	 Log_error 1 "!! TAPE PROBLEM: `date +'%H:%M:%S %d/%m/%y'`"
	else
	 echo " ---------------------------------------------------"| $TLOG
	 echo " Backup tape ejected at `date +'%H:%M:%S %d/%m/%y'` " | $TLOG
	 echo " ---------------------------------------------------"| $TLOG
	fi
 fi

 echo "==============================================================="| $TLOG
 echo "Oracle database backup ended on $HOST: `date '+%X %m/%d/%Y'`"   |$TLOG
 echo "==============================================================="| $TLOG

 #Append current log to permanent log, then keep it at 10000 lines

 cat $LOG >> $PERMLOG
 tail -10000 $PERMLOG >$PERMLOG.$X
 mv $PERMLOG.$X $PERMLOG

 Mail_dba
esac

#rm -f $TMP/$X.*              #Remove temporary logs

#End of oraback.sh
