#!/bin/sh
#
# Rcs_ID="$RCSfile: rclogs.sh,v $"
# Rcs_ID="$Revision: 1.1 $ $Date: 2000/06/30 18:00:40 $"
#
# Copyright (c) 1998 Curtis Preston curtis@colltech.com
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 2 of the License, or (at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; wit implied warranty of MERCHANTABILITY or 
# FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
# more details.
#
# For a copy of the license, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139
#
##########################################################################

########################################
#Site-specific configuration section

DEBUG=N                               #Set to Y to see DEBUG information
[ "$DEBUG" = Y ] && set -x
INFTAB=/informix/etc/inftab           #Set to full path of inftab file
COMPRESS='compress -f'                #Can get to gzip or pack
COMPSUFF='Z'                          #.gz if using gzip, .z if using pack
BINDIR=/informix/local/bin            #loc. of config.guess and localpath.sh
THRESHHOLD=75                         #Warn if ARCHIVEDEST is fuller than this
TMP=/var/tmp
INFORMIXDIR=`grep  "^$ONCONFIG:" $INFTAB | awk -F':' '{print $2}'`
ONCONFIGFILE="$INFORMIXDIR/etc/$ONCONFIG"
ARCHIVE_DEST=`grep  "^$ONCONFIG:" $INFTAB | awk -F':' '{print $4}'`
INFORMIXSERVER=`grep  "^DBSERVERNAME" $ONCONFIGFILE| awk '{print $2}'`
LOGDIR=$INFORMIXDIR/logs
LOG_FILE=$LOGDIR/rclogs.$INFORMIXSERVER.out
PATH="$INFORMIXDIR/bin:$PATH"
DBAS=`grep "^$ONCONFIG:" $INFTAB | awk -F':' '{print $10}'`
HOST=`hostname|awk -F. '{print $1}'`
[ -n "$HOST" ] || HOST=`uname -n|awk -F. '{print $1}'` 
[ -n "$HOST" ] || exit 1

#These next two values are used for the Cleanup and Desperate_cleanup functions
#Due to the way 'find' works, the values start with 0.  A 'mtime +0' will 
#find files older than 24 hours, 1=48 hrs, 2=72, 3=96, 4=120, 6=1 week, 
#12=wks, etc. 
#
#REG_OLD is used by the regular Cleanup function.  It specifies how old a 
#cont. backup file should be before it is normally deleted.  Default value 
#is 2, meaning files older than 72 hours will be deleted.  The GOAL would 
#be have a big enough ARCHIVE_DEST filesystem that you could leave enough 
#continuous backups for two cycles of full archives.  
#
#For example, if you are doing a level 0 archive once a week, the best 
#thing would be if #you could leave TWO WEEKS of archives on line.  
#( i.e. REG_OLD=12 )

REG_OLD=2  #MTIME value for normal cleanup.

#DESP_OLD is used by the Desperate_cleanup function.  This function is only 
#called if the filesystem is over $THRESHHOLD% full.  Do not specify a value 
#less than 1, since this doing so will remove files that are just over 24 
#hours old.  Such files may NOT be backed up to tape yet!  Therefore, the 
#most "desperate" you should get is "1"
#Normally, REG_OLD should be set much higher than 2, and DESP_OLD should be 
#set to two LESS than what you set REG_OLD to.  (That would delete two extra 
#days of logs.)

DESP_OLD=1  #MTIME value for desperate cleanup.  1=48 hrs, etc.

#End site-specific configuration section
########################################

export DEBUG INFTAB COMPRESS COMPSUFF BINDIR THRESHHOLD TMP INFORMIXDIR 
export ONCONFIGFILE ARCHIVE_DEST INFORMIXSERVER LOG_FILE PATH LOGDIR DBAS 
export REG_OLD DESP_OLD

#Define functions for tb/ontape section

Backup_the_backup()
{
[ "$DEBUG" = Y ] && set -x

BACKUP_FILE="$LTAPEDEV.$DATE.$COMPSUFF"

#EXAMPLES
#cp $BACKUP_FILE /nfs/filesystem/$BACKUP_FILE #Copy it to another system
#tar cf /dev/rmt/0mn $BACKUP_FILE             #Append it to a special tar tape
#save $BACKUP_FILE                            #Back it up with NetWorker

}

Check_file_system()
{
USED=`$DF $ARCHIVEDEST |tail -1|awk '{print $5}'`
AMESS="WARNING! $ARCHIVEDEST is getting full!" export AMESS

if [ "$USED" -gt $THRESHHOLD ] ; then
 Desperate_cleanup
 NEW_USED=`$DF $ARCHIVEDEST |awk '{print $5}'`
 echo "
 WARNING!  $ARCHIVEDEST is over ${THRESHOLD}% full!  ( It is ${USED}%. )
 It is still ${NEW_USED}% full after running the Desperate_cleanup function.
 (You should consider moving $ARCHIVEDEST to a bigger file system.)
 " >$TMP/$X.Mail
 $L_MAIL $L_S "$AMESS" $DBAS < $TMP/$X.Mail
fi

}

Cleanup()
{

#This function is run every time this script is run

[ "$DEBUG" = Y ] && set -x
#Remove files older than 72 hours
find . -name "$LTAPEDEV.*" -mtime +$REG_OLD -exec rm {} \;  
}

Desperate_cleanup()
{

#This function is run only if ARCHIVEDEST is more than $THRESHHOLD% full
#DO NOT lower this lower than 'mtime +1,' or else you may remove files 
#before they get backed up to tape!

[ "$DEBUG" = Y ] && set -x
find . -name "$LTAPEDEV.*" -mtime +$DESP_OLD -exec rm {} \; #Remove files older than 48 hours
}

Get_config()
{
[ "$DEBUG" = Y ] && set -x

 if [ -d $ARCHIVE_DEST ] ; then
  LTAPEDEV="$ARCHIVE_DEST/$INFORMIXSERVER.log"
 else
  if [ -c $ARCHIVE_DEST ] ; then
   LTAPEDEV=$ARCHIVE_DEST
  else
   echo "$ARCHIVE_DEST is neither a directory or a tape drive!"
   return 1
  fi
 fi

 $INFORMIXDIR/bin/$ONMONITOR ${INSTANCE} <<EOF >/dev/null
lt			$LTAPEDEV                       ee
EOF

 LTAPEDEV=`grep '^LTAPEDEV' $ONCONFIGFILE |awk '{print $2}'`

 if [ -z "$LTAPEDEV" ] ; then
  echo " $INFORMIXSERVER LTAPEDEV is not defined in $ONCONFIG."
  return 1
 fi

 if [ -c "$LTAPEDEV" ] ; then
  if [ "$LTAPEDEV" != '/dev/null' ] ; then
   DEVICE_TYPE="DISK"
  fi
 else
  DEVICE_TYPE="DISK"
 fi
 return 0
}

Setup_ltapedev()
{
[ "$DEBUG" = Y ] && set -x
 if [ $LTAPEDEV != '/dev/null' ] ; then
  touch $LTAPEDEV && chmod 660 $LTAPEDEV && \
  chgrp informix $LTAPEDEV && return 0
 else
  return 0
 fi
 echo "    FAILED to create $INFORMIXSERVER LTAPEDEV ($LTAPEDEV)."
 return 1
}

Perform_ontape()
{
[ "$DEBUG" = Y ] && set -x
 echo " Starting Continuous Backups to ($LTAPEDEV) ... "

 #echo ' ' | $ONTAPE -c |head -100 > $LOG_FILE &
 echo ' ' | $ONTAPE -c > $LOG_FILE &
 PID=$!
 echo >>$LOG_FILE
 echo "PID $PID" > $LOG_FILE.PID

 sleep 10

 PSENTRY=`$PS |grep "$PID.*$ONTAPE .c"|grep -v grep`

 if [ -n "$PSENTRY" ] ; then
  return 0
 else
  echo " FAILED to start Continuous Logging for $INFORMIXSERVER"
  return 1
 fi
}

Compress_file()
{
[ "$DEBUG" = Y ] && set -x
 if [ -s $LTAPEDEV ] ; then
  DATE=`date '+%Y.%m.%d.%H.%M.%S'|sed 's/:/./g'`
  echo "   Compressing $INFORMIXSERVER logfile to $LTAPEDEV.$DATE ..."

  mv $LTAPEDEV $LTAPEDEV.$DATE && \
  $COMPRESS $LTAPEDEV.$DATE && \
  test -s $LTAPEDEV.$DATE.$COMPSUFF && return 0
  echo "   FAILED to compress $INFORMIXSERVER Logging File."
  return 1
 else
  if [ "$LTAPEDEV" = '/dev/null' ] ; then
   echo "   !!! LOGGING ON $INFORMIXSERVER IS SET TO /dev/null !!!"
   echo "   !!! DATABASE WILL ONLY RECOVER UP TO LAST NIGHT'S ARCHIVE !!!"
  else
   echo "   Current $INFORMIXSERVER Logging File ($LTAPEDEV) is empty."
  fi
  return 0
 fi
}

Check_tape()
{
[ "$DEBUG" = Y ] && set -x

 #This section is 99% superfluous.  If Informix is running right, we will get
 #all this information from it.  However, in the remote chance that there is
 #an ontape -c process out there that this script started, and is no longer
 #connected to an Informix instance, this will end up finding it.

 if [ -f $LOG_FILE.PID ] ; then
   PID=`grep "^PID" $LOG_FILE.PID | awk '{print $2}'`
 else
   PID=''
 fi

 if [ -z "$PID" ] ; then
  PID=98765432123456789
 fi

 case $VERSION in
  4*|5*)
   INFPID=`$ONSTAT -u | grep 'A---M.*informix' | awk '{print $3}' `
  ;;

  6*|7*|8*)
   SESSION=`$ONSTAT -u | grep 'P..M.*informix' | awk '{print $3}'`
   if [ -n "$SESSION" ] ; then
    INFPID=`$ONSTAT -g ses | grep "^$SESSION" | awk '{print $4}'`
   else
    INFPID=''
   fi
  ;;
 esac

 if [ -n "$INFPID" ] ; then
  INFENTRY=`$PS |grep "$INFPID.*$ONTAPE .c"|grep -v grep` 
 else
  INFENTRY=""
 fi

 PSENTRY=`$PS |grep "$PID.*$ONTAPE .c"|grep -v grep` 

 if [ -n "$INFENTRY" -a -n "$PSENTRY" ] ; then
  PSENTRY=$INFENTRY
  PID=$INFPID
  echo "PID $INFPID $SESSION" > $LOG_FILE.PID
 fi

 if [ -n "$PSENTRY" ] ; then
  echo "   The following '$ONTAPE -c' process is running for $INFORMIXSERVER"
  echo "   $PSENTRY"
  return 0
 else
  echo "   There are now no '$ONTAPE -c' processes running for $INFORMIXSERVER."
  return 1
 fi
}

Kill_ontape()
{
[ "$DEBUG" = Y ] && set -x
 if [ -n "$PSENTRY" ] ; then
  echo $N "   Closing the current log ($ONMODE -l;$C "
  $ONMODE -l
  echo "$ONMODE -c)"
  $ONMODE -c
 
  sleep 5
 
  PID=`echo "$PSENTRY" | awk '{print $2}'`

  echo "   Stopping $INFORMIXSERVER '$ONTAPE -c' pid $PID"

  #If we're running Informix 6 or 7, we can use the onmode -z command
  case $VERSION in
  6*|7*)
   $ONMODE -z $SESSION && \
   sleep 10
  ;;
  esac

  [ -n "`$PS |grep \"$PID.*$ONTAPE .c\"|grep -v grep`" ] && \
  kill $PID && \
  sleep 10 && \
  [ -n "`$PS |grep \"$PID.*$ONTAPE .c\"|grep -v grep`" ] && \
  kill -9 $PID && \
  sleep 10

  ONTAPE_THERE=`$ONSTAT -u | grep 'P..M.*informix' `
  while [ -n "$ONTAPE_THERE" ] ; do
   echo "   Waiting for '$ONTAPE -c' to stop"
   sleep 5
   ONTAPE_THERE=`$ONSTAT -u | grep 'P..M.*informix' `
  done

 fi
 return 0
}

Sleep_loop()
{
[ "$DEBUG" = Y ] && set -x
 PSCHECK=`$PS | grep "$PID.*$ONTAPE .c"|grep -v grep`
 while [ -n "$PSCHECK" ] ; do
  sleep 5
  PSCHECK=`$PS | grep "$PID.*$ONTAPE .c"|grep -v grep`
 done
}

Change_ltapedev()
{
[ "$DEBUG" = Y ] && set -x
 echo " Unloading tape ... "
 $L_MT $L_F $LTAPEDEV $L_OFFLINE
 echo " Please remove tape on $HOST: $LTAPEDEV and"
 echo " load tape for $INFORMIXSERVER continuous logging."
 echo " Press <Return> when it's ready or <Ctrl-C> to stop."
 read READY
 $L_MT $L_F $LTAPEDEV $L_REWIND
 while [ $? -ne 0 ]
 do
  echo " Please remove tape on $LTAPEDEV on $HOST and"
  echo " load tape for $INFORMIXSERVER continuous logging."
  echo " Press <Return> when it's ready or <Ctrl-C> to stop."
  read READY
  $L_MT $L_F $LTAPEDEV $L_REWIND
 done
 return 0
}

Show_log_range()
{
[ "$DEBUG" = Y ] && set -x
 if [ -s $LOG_FILE ] ; then
  sed '1,/^Please label this/d' $LOG_FILE
  DATE=`date '+%Y.%m.%d.%H.%M.%S'|sed 's/:/./g'`
  echo " Please label the tape with above log number range."
 fi
}

##############################################################################
# MAIN
##############################################################################


[ "$DEBUG" = Y ] && set -x

if [ ! -f "$INFTAB" ] ; then
 echo " $INFTAB doesn't exist."
 exit 1
fi

if [ $# -lt 1 ] ; then
 echo " Missing Argument!"
 echo " Usage $0 [stop|start|stop-start]"
 exit 1
fi

if [ -z "$ONCONFIG" ] ; then
 echo " Missing ONCONFIG environment variable!"
 exit 1
fi

[ -d "$LOGDIR" ] || mkdir $LOGDIR
[ -w "$LOGDIR" ] || chmod 775 $LOGDIR

cp /dev/null $LOG_FILE
if [ $? -gt 0 ] ; then
 echo " Couldn't create $LOG_FILE"
 exit 1
fi

if [ ! -d "$INFORMIXDIR" ] ; then
 echo $N " ERROR found in $INFTAB for $ONCONFIG:$C"
 echo " INFORMIXDIR $INFORMIXDIR doesn't exist."
 exit 1
fi

if [ ! -f "$ONCONFIGFILE" ] ; then
 echo $N " ERROR found in $INFTAB for $ONCONFIG:$C"
 echo " ONCONFIGFILE $ONCONFIGFILE doesn't exist."
 exit 1
fi

if [ ! -f "$BINDIR/config.guess" ] ; then
 echo $N " ERROR found in $INFTAB for $ONCONFIG:$C"
 echo " config.guess file: $BINDIR/config.guess doesn't exist."
 exit 1
fi

if [ ! -f "$BINDIR/localpath.sh" ] ; then
 echo $N " ERROR found in $INFTAB for $ONCONFIG:$C"
 echo " localpath.sh script: $BINDIR/localpath.sh doesn't exist."
 exit 1
fi

if [ ! -d "$ARCHIVE_DEST" ] ; then
 echo $N " ERROR found in $INFTAB for $ONCONFIG:$C"
 echo " ARCHIVE_DEST $ARCHIVE_DEST doesn't exist."
 exit 1
fi

X=$$
ACTION=`echo $1 |tr '[a-z]' '[A-Z]'`

                        #Setup log and global variables
L_MAIL=mail ; L_S='' ; MESS='' ; export L_MAIL L_S MESS

if [ -f $BINDIR/config.guess ] ; then
 LOCAL_OSnR=`$BINDIR/config.guess`
 export LOCAL_OSnR
else
 Log_error 1 "NO $BINDIR/config.guess - ABORTING BACKUP" exit
fi

. $BINDIR/localpath.sh

MESS="INFORMIX CONTINUOUS BACKUP ERROR!" ; export MESS

#Make sure that ARCHIVEDEST has some space left!

Check_file_system

if [ -f $INFORMIXDIR/bin/onstat ] ; then
 ONMONITOR=onmonitor
 ONTAPE=ontape
 ONSTAT=onstat
 ONMODE=onmode
else
 ONMONITOR=tbmonitor
 ONTAPE=tbtape
 ONSTAT=tbstat
 ONMODE=tbmode
fi
 
VERSION=`$ONSTAT - |grep " Version "| sed '/^$/d'|awk '{print $3}'\
 |awk -F'.' '{print $1}'`

ps -ef >/dev/null && PS='ps -ef' || PS='ps -aux'

export ONMONITOR ONSTAT ONTAPE ONMODE VERSION PS

#Continuous backups will not start if db is not online
#Do an onstat and grep for Read-Only or On-Line

case $ACTION in
 STOP )       DOING=Stopping              ;;
 STOP-START ) DOING='Stopping & Starting' ;;
 START )      DOING=Starting              ;;
esac

echo " $DOING continuous backups for $INFORMIXSERVER..."

echo $N "  Waiting for $INFORMIXSERVER to report 'On-Line' or 'Read-Only'.$C"
 
DB_UP=`$ONSTAT - |egrep -c 'On.Line|Read.Only'`

CT=0
while [ $DB_UP = 0 ] ; do
 sleep 5
 echo $N ".$C"
 DB_UP=`$ONSTAT - |egrep -c 'On.Line|Read.Only' `
 CT=`expr $CT + 1`
 if [ $CT -eq 30 ] ; then
  DB_UP=1
  echo 
  echo $N " $INFORMIXSERVER did not report \'On-Line\' or \'Read-Only\'$C"
  echo " after 60 seconds."
  echo " $0 will continue on without waiting anymore."
  echo " WARNING: CONTINUOUS BACKUPS MAY NOT START PROPERLY."

  #This has to be done, else one bad instance will hang the startup
  #of the whole system

 fi
done
echo

#Perform the tb/ontape continuous backup section
BACKUP="$ONTAPE -c"

ERROR=NO

#
#Routines inside this loop
#RETURNS 1 when error happens
#Check_tape returns 0 when Continuous Logging
#(ontape -c) is up and running
#

Get_config || ERROR=YES 

#
#If 'ontape -c' is running, stop it,
#('ontape -c' takes a few seconds to finish).
#Wait for 5 seconds, if 'ontape -c' still
#running, echo error message.
#

echo "  Stopping continuous backups (if running),"
echo "  and moving & compressing backup file."

#Check_tape looks for an ontape -c process.  If it finds it, it continues
#Then we try to stop it with onmode, and sleep for a while
#Then we check again for it.  If it's still there, we kill it.
#Then we wait some more.  If it's STILL there, we kill it with a -9 (should never get here)

Check_tape && \
 Kill_ontape && \
 Sleep_loop && \
 Check_tape && \
 echo " FAILED to kill $INFORMIXSERVER '$ONTAPE -c' pid $PID" && \
 ERROR=YES && \
 continue

#Check tape returns a 0 if there are processes running, therefore it will 
#continue to Kill_ontape and echo "FAILED..." if there are still processes 
#running.  If there weren't any to begin with, or if we stop them with 
#Kill_ontape, then we go on.

case "$DEVICE_TYPE" in
 TAPE)
  Show_log_range
  Change_ltapedev || ERROR=YES
 ;;
 DISK)
  #Before we do anything else, clean out oldfiles
  Cleanup         || ERROR=YES
  Compress_file   || ERROR=YES
  Setup_ltapedev  || ERROR=YES
 ;;
esac

if [ "$ACTION" = "STOP" ] ; then
 if [ "$ERROR" = "NO" ] ; then
  echo " Continous backups stopped for $INFORMIXSERVER"
  exit 0
 else
  echo " Continous backups NOT STOPPED for $INFORMIXSERVER!"
  exit 1
 fi
fi

Perform_ontape || ERROR=YES

#
# We should have 'ontape -c' running
# after Perform_ontape.

Check_tape || ERROR=YES


if [ "$ERROR" = "NO" ] ; then     
 echo " Continuous Backups for $INFORMIXSERVER are started."
 exit 0
else
 echo
 echo " CONTINUOUS BACKUPS DID NOT START FOR INFORMIXSERVER $INFORMIXSERVER!!" 
 exit 1
fi

#This function can be customized to run another backup of the logical log
#to another location (another disk, another tape, etc.)
#The default function only removes files older than 72 hours
#It's at the end, so we restart ontape FIRST, THEN backup and cleanup

if [ "$DEVICE_TYPE" = DISK ] ; then
 Backup_the_backup
fi

#Check file system again now that we're done

Check_file_system
