#!/bin/perl
#
# fpserver.cgi - Firepass server
# By Alex Dyatlov <alex@gray-world.net>
# Download the latest Firepass version at http://gray-world.net
# This program is distributed under the terms of the GNU General Public License v2.0
# See the file COPYING for details
#
# VERSION 1.0.2a
#
# May, 2003
#
# ----------------------------------- Configuration section - begin
my $inoutdir = "/var/tmp/inout";# Path to existing data exchange directory
my $logdir = "/var/tmp/log";	# Path to existing log directory
my $logf = "fpserver.log";	# Log file name
my $inm = "_in.dat";		# Echange files masks
my $outm = "_out.dat";
my $lockm = ".lock";
my $log = "no";			# ("no" | "yes") Create log file?
my $useacl = "yes";		# ("no" | "yes") Use access list?
my @ALLOW = qw (
	127.0.0.1
	111.222.33.4
);				# Access list, one IP for each line
my $initdelay = .3;		# New connection initialization delay (sec)
my $checkdelay = .3;		# Data exchange files check delay
my $faketype = "text/html";
my $fakepage = qq (
<html>
<body></body>
</html>
);				# HTML page for browsers requests
my $readsize    = 20480;        # Socket read buffer
# ----------------------------------- Configuration section - end

use strict;
use IO::Socket;
use Fcntl ':flock';

sub forkmanager(@); sub dataready(@); sub closeses(@); sub msleep(@);
sub _read(@); sub _write(@); sub _logerror(@); sub _log(@);

$| = 1;

my $session	= $ENV{HTTP_X_SESSION};
my $counter	= $ENV{HTTP_X_COUNTER};
my $rip		= $ENV{REMOTE_ADDR};
my $rport	= $ENV{REMOTE_PORT};

unless (defined($session)) {
	_log("parent", "n/a", "connection from $rip:$rport without session id");
	print "Content-Type: $faketype\r\n\r\n";
	print "$fakepage";
	exit 0;
}

if ($useacl eq "yes") {
	my $al = 0;
	foreach my $ip (@ALLOW) {
		$al = 1
			if ($rip =~ /$ip/);
	}
	_logerror("parent", $session, "no access list record for $rip")
		unless ($al);
}

my $fin = "$inoutdir/$session$inm";
my $fout = "$inoutdir/$session$outm";

if ($counter == 1) {
	my $host  = $ENV{HTTP_X_HOST};
	my $port  = $ENV{HTTP_X_PORT};
	my $proto = $ENV{HTTP_X_PROTO};
	_logerror("parent", $session, "redirect host, port or protocol missed")
		if ($host !~ /\S+/ || $port !~ /\d+/ || ($proto !~ /tcp/i && $proto !~ /udp/i));
	_log("parent", $session, "connection from $rip:$rport; redirecting to $host:$port/$proto");
	_logerror("parent", $session, "fail to create files in $inoutdir/ directory: $!")
		unless (open(IH, "> $fin") && open(OH, "> $fout"));
	close(IH);
	close(OH);
	$SIG{CHLD} = "IGNORE";
	my $child = forkmanager($session, $host, $port, $proto);
	_logerror("parent", $session, "fail to fork Connection Manager for $rip")
		if ($child == undef);
	msleep($initdelay)
		if ($initdelay > 0);
}

if ($ENV{HTTP_X_CONNECTION} eq "close") {
	msleep($checkdelay)
		while (-s $fout > 0);
	closeses();
	_log("parent", $session, "connection from $rip:$rport closed");
	print "Content-Length: 0\r\n".
		"X-Connection: close\r\n".
		"\r\n";
	exit 0;
}

unless (-e $fin && -e $fout) {
	print "Content-Length: 0\r\n".
		"X-Connection: close\r\n".
		"\r\n";
	_log("parent", $session, "connection from target closed");
	exit 0;
}

if ($ENV{CONTENT_LENGTH} > 0) {
	read(STDIN, my $buf, $ENV{CONTENT_LENGTH});
	my $bw = _write($fout, $buf);
	_logerror("parent", $session, "fail write to $fout")
		if ($bw == undef);
}

my $size = -s $fin;
print "Content-Type: application/octet-stream\r\n".
      "X-Connection: alive\r\n";
if ($size == 0) {
	print "Content-Length: $size\r\n".
	      "\r\n";	
}
if ($size > 0) {
	my $buf = _read($fin);
	print "Content-Length: ".(length($buf))."\r\n".
	      "\r\n";	
	print $buf;
}

exit 0; # main() end

sub forkmanager(@) {
	(my $session, my $host, my $port, my $proto) = @_;
	my $f = fork();
	if ($f == 0) {
		close(STDIN);
		close(STDOUT);
		close(STDERR);
		my $SOCKET = IO::Socket::INET->new(
			PeerAddr => $host,
			PeerPort => $port,
			Proto    => $proto,
			Type     => SOCK_STREAM
		);
		if ($SOCKET == undef) {
			_log("child", $session, "connection fail to $host:$port/$proto");
			msleep($checkdelay)
				while (-s $fin > 0);
			closeses();
			exit 0;
		}
		my $stotal = my $ctotal = 0;
		while (1) {
			unless (-e $fin && -e $fout) {
				_log("child", $session, "connection from client closed; $stotal".
					" bytes received from target / $ctotal bytes sent");
				shutdown($SOCKET, 2);
				exit 0;
			}
			my $d = dataready($SOCKET, $checkdelay);
			my $size = -s $fout;
			if ($d == $SOCKET) {
				my $bytes = sysread($SOCKET, my $buf, $readsize);
				if ($bytes == 0) {
					_log("child", $session, "connection from target closed; $stotal".
						" bytes received from target / $ctotal bytes sent");
					msleep($checkdelay)
						while (-s $fin > 0);
					closeses();
					exit 0;
				}
				$stotal += $bytes;
				_write($fin, $buf);
			}
			if ($size > 0) {
				my $buf = _read($fout);
				$ctotal += length($buf);
				print $SOCKET $buf;
			}
		}
	}
	return $f;
}

sub dataready(@) {
	my @s = @_;
	my $to = $s[$#s];
	$#s--;
	my $rin = "";
	foreach my $d (@s) {
		vec($rin, fileno($d), 1) = 1;
	}
	my $nfound = select(my $rout = $rin, undef, my $eout = $rin, $to);
	foreach my $d (@s) {
		return $d
			if (vec($eout, fileno($d), 1) || vec($rout, fileno($d), 1));
	}
	return 0;
}

sub closeses(@) {
	unlink($fin) if (-e $fin);
	unlink($fout) if (-e $fout);
	unlink("$fin$lockm") if (-e "$fin$lockm"); 
	unlink("$fout$lockm") if (-e "$fout$lockm");
	return;
}

sub msleep(@) {
	my $i = shift;
	select(undef, undef, undef, $i);
	return;
}

sub _read(@) {
	my $file = shift;
	my $buf;
	open(RLOCK, "> $file$lockm")
		or return undef;
	flock(RLOCK, LOCK_EX);
	open(RH, "+< $file")
		or return undef;
	flock(RH, LOCK_EX);
	my $size = -s $file;
	my $br = read(RH, $buf, $size);
	open(RH, "> $file")
		or return undef;
	close(RH);
	close(RLOCK);
	return $buf;
}

sub _write(@) {
	my ($file, $buf) = @_;
	open(WLOCK, "> $file$lockm")
		or return 0;
	flock(WLOCK, LOCK_EX);
	open(WH, ">> $file")
		or return 0;
	flock(WH, LOCK_EX);
	print WH $buf;
	close(WH);
	close(WLOCK);
	return 1;
}

sub _logerror(@) {
	my ($src, $session, $s) = @_;
	print "Content-Length: 0\r\n".
		"X-Connection: close\r\n".
		"\r\n"
		if ($src eq "parent");
	_log($src, $session, "error: $s");
	exit 0;
}

sub _log(@) {
	my ($src, $session, $s) = @_;
	return if ($log ne "yes");
	(my $sec, my $min, my $hour, my $day, my $month, my $year) = (localtime)[0,1,2,3,4,5];
	$year += 1900;
	$month++;
	open(FH, ">> $logdir/$logf");
	flock(FH, LOCK_EX);
	my $date = sprintf("%4d/%02d/%02d [%02d:%02d:%02d]", $year, $month, $day, $hour, $min, $sec);
	print FH "$date $src [\#$session] $s\n";
	close(FH);
	return;
}
